/* Author: Helen Wollan <h.e.wollan@sms.ed.ac.uk>
 * Updated: Thurs Sept 02 2004 by Helen Wollan
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.ip2.map;

import java.awt.Color;
import java.text.*;
import java.util.*;
import java.lang.reflect.Array;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.awt.Component;

import com.bbn.openmap.Layer;
import com.bbn.openmap.*;
import com.bbn.openmap.layer.OMGraphicHandlerLayer;
import com.bbn.openmap.omGraphics.OMGraphic;
import com.bbn.openmap.omGraphics.OMGraphicList;
import com.bbn.openmap.omGraphics.OMPoly;
import com.bbn.openmap.omGraphics.OMText;
import com.bbn.openmap.proj.Projection;
import com.bbn.openmap.omGraphics.event.*;
import com.bbn.openmap.event.*;
import com.bbn.openmap.geo.Geo;
import com.bbn.openmap.gui.*;
import com.bbn.openmap.event.*;

import com.bbn.openmap.layer.*;
import com.bbn.openmap.omGraphics.*;

import ix.util.Util;
import ix.util.Parameters;
import ix.iface.util.CatchingActionListener;
import ix.icore.domain.PatternAssignment;

import ix.util.*;
import ix.icore.domain.*;
import ix.iface.util.*;

import ix.icore.process.event.*; 
import ix.icore.IXAgent;
import ix.util.lisp.*;
import ix.ip2.*;
import ix.ip2.StateViewMap;

public class IxSectorLayer extends OMGraphicHandlerLayer implements ProcessStatusListener, MapMouseListener {

    private OMGraphicList omgraphics; 
    private Map map;
    private Projection proj;
    private Ip2 agent;
    private Vector pointslist;
    private OMPoly currpoly;
    private Vector pointlist;
    private MapBean mapbean;
    private LatLonPoint point;
    private Hashtable sectorlist;
    private Hashtable searcherhash;
    private Hashtable sectorpoints;
    private Hashtable probhash;
    private Hashtable statushash;
    private Hashtable noteshash;
    private Hashtable notelist;
    private Hashtable widthhash;
    private boolean build;
    private JButton buildButton = new JButton();
    private Color sectorColor = Color.black;
    private int lineType = OMGraphic.LINETYPE_STRAIGHT; // OMGraphic.LINETYPE_STRAIGHT
    private JPopupMenu menu;
    private JMenuItem prob, status, searcher, notes, track;
    private JMenu searcherlist;
    private JTextArea newnote;
    private Box box;
    private JFrame noteframe;
    private String tempsector;
    private int sectornumber = 1;
    

    public IxSectorLayer() {
        super();       
		omgraphics = (OMGraphicList)getList();
		
		//initialise the hashtables to keep track of the various details of the sector
		sectorlist = new Hashtable();
		searcherhash = new Hashtable();
		sectorpoints = new Hashtable();
		probhash = new Hashtable();
		statushash = new Hashtable();
		noteshash = new Hashtable();
		widthhash = new Hashtable();
		
		agent = (Ip2) IXAgent.getAgent();
		(agent.getModelManager()).addProcessStatusListener(this);
		
		// initialise some global variables:
		// the current list of points for a sector while building
		pointlist = new Vector();
		// the current sector that is being built
		currpoly = new OMPoly();
		// boolean for whether user currently building sectors
		build = false;
    }

	// initialise the properties of the layer
    public void setProperties(String prefix, Properties props) {
        super.setProperties(prefix, props);
        
    }
    
    // set the MapBean (called from MapTool)
    public void setMapBean(MapBean mb) {
    	mapbean = mb;
    }
    
    // simple method to send a constraint to the IX agent.  Copied from WorldStateLayer
    private void sendConstraint(PatternAssignment pa){
        Vector v = new Vector();
        v.add(pa);
        Constraint c = new Constraint(Symbol.intern("world-state"),Symbol.intern("effect"),v);
        ((Ip2ModelManager)agent.getModelManager()).addConstraint(c);
    }    
	
	// render the omgraphics on the map
    public void paint(java.awt.Graphics g) {
        omgraphics.render(g);
    }
    
	// repaint the omgraphics when the projection is changed by the user
    public void projectionChanged(ProjectionEvent pe) {
        Projection oldProj = proj;
        proj = setProjection(pe);
        if (proj != null) {
            getList().generate(proj);
            repaint();
        }
        else
            proj = oldProj;
    
        fireStatusUpdate(LayerStatusEvent.FINISH_WORKING);
    }   

	// return the list of the omgraphics
    public OMGraphicList getList() {
        OMGraphicList list = super.getList();
        if (list == null) {
            list = new OMGraphicList();
            super.setList(list);
        }
        return list;
    }

	// remove all data contained in the sector layer
    public void reset(){
        for(int i=omgraphics.size()-1;i>=0;i--){
            omgraphics.removeOMGraphicAt(i);
        }
        getList().generate(proj);
        repaint();
        // clear all hashtables:
        sectorlist.clear();
        searcherhash.clear();
        probhash.clear();
        statushash.clear();
        noteshash.clear();
        sectorpoints.clear();
    }

    // ProcessStatusListener 
    // This reads in the constraints whenever they are changed and performs the appropriate
    // operations on them.

    public void stateChange(ProcessStatusEvent event, Map delta) {
    	map = delta;
    	String sector;
    	String name;

		try{
	    	for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
				Map.Entry e = (Map.Entry)i.next();
				LList pattern = (LList)e.getKey(); 
	
				LList property = ((LList)pattern.clone()).delete(pattern.get(1));
				
				// the constraint is a sector constraint: this contains the verticies of
				// a sector polygon.
				if( ((property.get(0)).toString()).equals("sector") ) {
					Vector points = new Vector();
					sector = (pattern.get(1)).toString(); // the sector name
		    		OMPoly path = null;
		    		LList value = (LList) e.getValue();
		    		float pointsF[] = new float[value.size()*2];
					
					// reading through the points and adding them to a float[] array and a 
					// vector for internal storage.
		    		for(int x=0;x<value.size();x++){
						LList point = (LList) value.get(x);
						pointsF[2*x] = Float.parseFloat((point.get(0)).toString());
						pointsF[2*x+1] = Float.parseFloat((point.get(1)).toString());
						points.add(new Float((point.get(0)).toString()));
						points.add(new Float((point.get(1)).toString()));
		    		}
		    		// add the vector of vertices to the hashtable under the sector name
					sectorpoints.put(sector, points);
					
					//check if the sector already exists and if so, modify the location of the OMPoly
		    		if (sectorlist.containsKey(sector)) {
						path = (OMPoly) sectorlist.get(sector);
						path.setLocation(pointsF,OMGraphic.DECIMAL_DEGREES);
		    		} else {
		    			// the sector is new so create a new OMPoly
						path = new OMPoly(pointsF,OMGraphic.DECIMAL_DEGREES,lineType);
						path.setLinePaint(sectorColor);	
						omgraphics.add(path);
						// add the OMPoly to the hashtable for storage
						sectorlist.put(sector,path);
		    		}
					
					getList().generate(proj);
					repaint();
					
	    		} else if( ((property.get(0)).toString()).equals("searcher") ) {
	    			// the constraint is the list of searchers for a sector
	    			
	    			sector = (pattern.get(1)).toString(); // the sector name
	    			Vector names = new Vector();
					
					// add the searcher names to a vector and put that vector in a hashtable
					// with the name of the sector as the key
	    			LList value = (LList) e.getValue();
	    			for (int j = 0; j < value.size(); j++) {
						name = value.get(j).toString();
						names.add(name);
					}
					searcherhash.put(sector, names);
					
	    		} else if (((property.get(0)).toString()).equals("probability")) {
	    			// the constraint is the probability of the sector
	    			// put the probability in a hashtable with the sector name as the key
	    			sector = (pattern.get(1)).toString();
	    			Double pvalue = (Double) e.getValue();
	    			probhash.put(sector, pvalue);
	    			
	    		} else if (((property.get(0)).toString()).equals("status")) {
	    			// the constraint is the status of the sector (a percentage complete)
	    			// put the status in a hashtable with the sector name as the key
	    			sector = (pattern.get(1)).toString();
	    			Double svalue = (Double) e.getValue();
	    			statushash.put(sector, svalue);
	    			
	    		} else if (((property.get(0)).toString()).equals("note")) {
	    			// the constraint is the list of notes on the sector
	    			sector = (pattern.get(1)).toString();
	    			Hashtable notes = new Hashtable();
	    			
	    			// get the notes out of the form ((date note) (date note)) to a simple vector
	    			// and add to the hashtable with sector name as the key
	    			LList value = (LList) e.getValue();
	    			for (int j = 0; j < value.size(); j++) {
	    				LList indnote = (LList) value.get(j);
	    				String date = indnote.get(0).toString();
	    				String note = indnote.get(1).toString();
	    				notes.put(date, note);
	    			}
	    			noteshash.put(sector, notes);
	    		} else if (((property.get(0)).toString()).equals("trackWidth")) {
	    			// the constraint is the track width of the sector
	    			// put the probability in a hashtable with the sector name as the key
	    			sector = (pattern.get(1)).toString();
	    			Double wvalue = (Double) e.getValue();
	    			widthhash.put(sector, wvalue);
	    		}
	    			
	    	}
		} catch(Exception exception) {
			System.err.println("Syntax error in sector "); 
			exception.printStackTrace();
		}
		
    }

	// method to deal with constraints being deleted from IX.
    public void stateDeletion(ProcessStatusEvent event, java.util.Map delta){
		for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
		    Map.Entry e = (Map.Entry)i.next();
	    	LList pattern = (LList)e.getKey(); 
	    	LList property = ((LList)pattern.clone()).delete(pattern.get(1));
	
	    	if( ((property.get(0)).toString()).equals("sector") ) {
	    		// if the sector is deleted then remove the OMPoly and all associated constraints:
	    		// probability, searcher, status, and notes
		        if (sectorlist.containsKey(pattern.get(1))) {
			    	omgraphics.remove((OMPoly)sectorlist.get(pattern.get(1)));
			    	sectorlist.remove(pattern.get(1));
		    		sectorpoints.remove(pattern.get(1).toString());
            		getList().generate(proj);
             		repaint();
				}
	    	} else if (((property.get(0)).toString()).equals("probability")) {
	    		// Just probability was deleted - remove from the hash table
		    	if (probhash.containsKey(pattern.get(1))) {
	    			probhash.remove(pattern.get(1));
	    		}
	    	} else if (((property.get(0)).toString()).equals("status")) {
	    		// Just the status was deleted - remove from the hash table
		    	if (statushash.containsKey(pattern.get(1))) {
	    			statushash.remove(pattern.get(1));
	    		}	
	    	} else if (((property.get(0)).toString()).equals("notes")) {
	    		// just the notes was deleted - remove from the hash table
		    	if (noteshash.containsKey(pattern.get(1))) {
	    			noteshash.remove(pattern.get(1));
		    	}
	    	}
    	}
    	getList().generate(proj);
        repaint();
    }

    public void newBindings(ProcessStatusEvent e, java.util.Map bindings){}
    public void statusUpdate(ProcessStatusEvent e){}
    
    
    // Mouse methods for MapMouseListener
    
    public MapMouseListener getMapMouseListener() {
    	return this;
    }
    
    public String[] getMouseModeServiceList() {
        String[] services = {SelectMouseMode.modeID};
        return services;    	
    }
    
    // action to be taken when the mouse is clicked on the Map Tool.
    public boolean mouseClicked(MouseEvent evt) {
    	int click = evt.getClickCount();
    	int buttonClick = evt.getButton();
    	
    	// get the coordinates of the mouse click
    	point = mapbean.getCoordinates(evt);
    	Float x = new Float(point.getLatitude());
    	Float y = new Float(point.getLongitude());
    	
    	// if the user right-clicks then bring up the appropriate popup menu
    	if (buttonClick == evt.BUTTON3) {
    		// check if the user right clicks in a sector
    		String sector = pointInSector(x.floatValue(), y.floatValue());
    		if (sector == null) {return true;} // user is not in a sector
    		
    		// create and display the appropriate menu for the sector
    		statusMenu(sector);
    		menu.show(evt.getComponent(), evt.getX(), evt.getY());
    		return true;
    	}
    	
    	// the user did not right click.  So check if the user is in build state and 
    	// verify the user leftclicked.
    	if (click == 1 && build && buttonClick == evt.BUTTON1) {
    		// this action is done whenever the user clicks
    		
    		// add the point to the vector and create a float array out of the vector
    		pointlist.add(x);
    		pointlist.add(y);
    		float[] points = new float[pointlist.size()];
    		for (int i = 0; i < pointlist.size(); i++) {
	    		points[i] = ((Float)pointlist.elementAt(i)).floatValue();
    			points[i+1] = ((Float)pointlist.elementAt(i+1)).floatValue();
    			i++;
    		}
    		
    		// set the current poly and add to the gui
    		currpoly = new OMPoly(points, OMGraphic.DECIMAL_DEGREES, OMGraphic.LINETYPE_STRAIGHT);
    		omgraphics.add(currpoly);
    		getList().generate(proj);
    		repaint();
    	} else if (click == 2 && build && buttonClick == evt.BUTTON1) { 
    		// if the user double clicks then verify the sector is closed and add to the constraints
    		
    		if (!(pointlist.elementAt(0)).equals(pointlist.elementAt(pointlist.size() - 2)) || !(pointlist.elementAt(1)).equals(pointlist.elementAt(pointlist.size() - 1))) {
    			// making the polygon sector closed by adding the start point to the end
    			pointlist.add(pointlist.elementAt(0));
    			pointlist.add(pointlist.elementAt(1));
    		}
    		
    		// need to make pointlist a LList of the form ((  ) (  ) ...) and send as a constraint
    		LList temppoints = Lisp.NIL;
    		LList temp;
    		for (int i = 0; i < pointlist.size(); i++) {
    			temp = Lisp.NIL;
    			temp = Lisp.cons(pointlist.elementAt(i+1), temp);
    			temp = Lisp.cons(pointlist.elementAt(i), temp);
    			i++;
    			temppoints = Lisp.cons(temp, temppoints);
    		}
    		pointlist.clear(); //time for a new sector - was a double click
    		
    		// sectors default with name searchx where x is an int starting at 1.
    		sendConstraint(new PatternAssignment(Lisp.list(Symbol.intern("sector"), Symbol.intern("search" + sectornumber)), temppoints));
    		sectornumber++;
    	}
    	
    	return true;
    }
    
    // other mouseEvent methods required for the MapMouseListener
    public boolean mouseDragged(MouseEvent evt) {return true;}
    public void mouseEntered(MouseEvent evt) {}
    public void mouseExited(MouseEvent evt) {}
    public void mouseMoved() {}
    public boolean mouseMoved(MouseEvent evt) {return true;}
    public boolean mousePressed(MouseEvent evt) {return true;}
    public boolean mouseReleased(MouseEvent evt) {return true;}
    
    
    // method to define the popup menu
    private void statusMenu(String sector) {
    	tempsector = sector;
		menu = new JPopupMenu();
		// allows the user to both see and modify the probability
		prob = new JMenuItem("Probability of Area");
		prob.addActionListener(this);
		// allows the user to both see and modify the status
		status = new JMenuItem("Status");
		status.addActionListener(this);
		// allows the user to both see and add notes to the sector
		notes = new JMenuItem("Display Notes");
		notes.addActionListener(this);
		// allows the user to add searchers to the sector
		searcher = new JMenuItem("Add Searcher");
		searcher.addActionListener(this);
		// allows the user to review the searchers assigned to the sector
		searcherlist = new JMenu("Searchers");
		// allows the user to both see and modify the track width
		track = new JMenuItem("Track Width");
		track.addActionListener(this);
		menu.add(prob);
		menu.add(status);
		menu.add(notes);
		
		// create menu items for each searcher assigned to the sector to be displayed
		// within the menu
		if (searcherhash.containsKey(sector)) {
			Vector names = (Vector)searcherhash.get(sector);
			for (int i = 0; i < names.size(); i++) {
				JMenuItem jmi = new JMenuItem((String)names.elementAt(i)); 
				searcherlist.add(jmi);
			}
			menu.add(searcherlist);
		}   	
		menu.add(searcher);
		menu.add(track);

    }
    
    
    // method to determine which sector, if any, a coordinate point is in.
    private String pointInSector(float pointLat, float pointLon) {
    	Enumeration sectors = sectorpoints.keys();
    	
    	// iterate through each sector and see if the point is in any of them
    	while (sectors.hasMoreElements()) {
    		String key = (String) sectors.nextElement();
    		Vector points = (Vector) sectorpoints.get(key);
    		float[] poly = new float[points.size()];
    		for (int i = 0; i < points.size(); i++) {
    			poly[i] = ((Float)points.elementAt(i)).floatValue();
    		}
    		if (pointInPolygon(poly, pointLat, pointLon)) {
    			return key; // the point is in a sector - return the sector name
    		}
    	}
    	
    	return null; // the point is not in any sector - return null
    }
    
    
    // code checks if point described by pointLat and pointLon is in the polygon polyVector
    // code modified from http://www.alienryderflex.com/polygon accessed 29 July 2004
    // returns true if the point is in the polygon, false otherwise
    private boolean pointInPolygon(float[] poly, float pointlat, float pointlon) {
    	int j = 0;
    	int k = 0;
    	int i = 0;
    	int polysize = poly.length / 2;
    	
    	int rcross = 0;
		int lcross = 0;
		boolean rstrad, lstrad;
		float x = 0;

    	float[] polylat = new float[polysize];
    	float[] polylon = new float[polysize];
    	for (i = 0; i < poly.length; i++) {
			polylat[k] = poly[i];
			polylon[k] = poly[i+1];
			k++;
			i++;
		} 
		
		for (i = 0; i < (polysize - 1); i++) {
			j++;
			if (j == polysize) j = 0;
			if (polylat[i] == pointlat && polylon[i] == pointlon) {
				return true; // point is a vertex
			}

			rstrad = (polylon[i] > pointlon) != (polylon[j] > pointlon);
			lstrad = (polylon[i] < pointlon) != (polylon[j] < pointlon);

			if (rstrad || lstrad) {
				x = pointlon * (float) ((float) (polylat[i] - polylat[j]) / (float) (polylon[i] - polylon[j])) - polylon[i] * (float) ((float) polylat[i] - polylat[j]) / (float)(polylon[i] - polylon[j]) + polylat[i];
    			if (rstrad && (x > pointlat)) {
      				rcross++;
    			}
    			if (lstrad && (x < pointlat)) {
      				lcross++;
    			}
  			}
		}	
		
		if ((rcross % 2) != (lcross % 2)) {
			return true; // on the edge
		} 
		if ((rcross % 2) == 1) {
  			return true; // inside the poly
		} else {
  			return false; // outside the poly, edge, or vertices
		}
    	
    }
    		
    // method required for the layer to have a properties window.  This allows the user
    // to determine if they are in build state or not.  The button changes accordingly.
    public Component getGUI() {
    	if (build == true) {
    		buildButton.setText("Finish Building");
    	} else {
    		buildButton.setText(" Build Sectors ");
    	}
    	buildButton.addActionListener(this);
    	return buildButton;
    }
    
    
    // actionlistener definitions for the properties and the popup menu
    public void actionPerformed(ActionEvent aevt) {
    	String cmd = aevt.getActionCommand();
    	if (cmd.equals(" Build Sectors ")) {
   			buildButton.setText("Finish Building");
   			build = true;
    	} else if (cmd.equals("Finish Building")) {
    		build = false;
    		buildButton.setText(" Build Sectors ");
    	} else if (cmd.equals("Probability of Area")) {
    		changeProbability();
    	} else if (cmd.equals("Status")) {
    		changeStatus();
    	} else if (cmd.equals("Display Notes")) {
    		displayNotes();
    	} else if (cmd.equals("Add Searcher")) {
    		addSearcher();
    	} else if (cmd.equals("Track Width")) {
    		changeWidth();
    	} else if (cmd.equals("OK")) {
    		addNote();
    	} else if (cmd.equals("Cancel")) {
    		noteframe.setVisible(false);
    	}
    }
    
    // brings up option pane allowing user to modify the probability for the sector
    // has error checking to ensure probability is between 0 and 100
    private boolean changeProbability() {
    	String p;
    	if (probhash.containsKey(tempsector)) {
    		p = ((Double)probhash.get(tempsector)).toString();
    	} else {
    		p = "0.0";
    	}
    	
    	String ans = (String) JOptionPane.showInputDialog(mapbean,"Change Probability\nProbability must be between 0 and 100", p);
    	while (ans != null) { 
    		Double newprob = new Double(ans);
    		if (Double.compare(newprob.doubleValue(), 100.0) <= 0 && Double.compare(newprob.doubleValue(), 0.0) >= 0) {
    			sendConstraint(new PatternAssignment(Lisp.list(Symbol.intern("probability"),Symbol.intern(tempsector)), new Double(ans)));
    			return true;
    		}
    		ans = (String) JOptionPane.showInputDialog(mapbean,"Change Probability\nProbability must be between 0 and 100", p);
    		
    	}
    	return true;
    }
    
    // brings up option pane allowing user to modify the status for the sector
    // has error checking to ensure status is between 0 and 100
    private boolean changeStatus() {
    	String s;
    	if (statushash.containsKey(tempsector)) {
    		s = ((Double)statushash.get(tempsector)).toString();
    	} else {
    		s = "0.0";
    	}
    	
    	String ans = (String) JOptionPane.showInputDialog(mapbean, "Change Status\nStatus must be between 0 and 100", s);
    	while (ans != null) {
    		Double newprob = new Double(ans);
    		if (Double.compare(newprob.doubleValue(), 100.0) <= 0 && Double.compare(newprob.doubleValue(), 0.0) >= 0) {
    			sendConstraint(new PatternAssignment(Lisp.list(Symbol.intern("status"),Symbol.intern(tempsector)), new Double(ans)));
    			return true;
    		}
    		ans = (String) JOptionPane.showInputDialog(mapbean, "Change Status\nStatus must be between 0 and 100", s);
    	}
    	return true;
    }
    
    // allows the user to add a searcher name to the sector
    private void addSearcher() {
    	Vector names;
    	if (searcherhash.containsKey(tempsector)) {
    		names = (Vector)searcherhash.get(tempsector);
    	} else {
    		names = new Vector();
    	}
    	
    	String ans = (String) JOptionPane.showInputDialog(mapbean, "Add Searcher");
    	if (ans != null) {
    		names.add(ans);
    		LList tempnames = Lisp.NIL;
    		for (int i = 0; i < names.size(); i++) {
    			tempnames = Lisp.cons(names.elementAt(i), tempnames);
    		}
    		sendConstraint(new PatternAssignment(Lisp.list(Symbol.intern("searcher"), Symbol.intern(tempsector)), tempnames));
    	}
    	
    }

    private boolean changeWidth() {
    	String w;
    	if (widthhash.containsKey(tempsector)) {
    		w = ((Double)widthhash.get(tempsector)).toString();
    	} else {
    		w = "5.0"; // the default
    	}
    	
    	String ans = (String) JOptionPane.showInputDialog(mapbean,"Change Track Width\nTrack Width is in kilometers", w);
    	if (ans != null) { 
    		Double newprob = new Double(ans);
    		sendConstraint(new PatternAssignment(Lisp.list(Symbol.intern("trackWidth"),Symbol.intern(tempsector)), new Double(ans)));
    		return true;
    	}
    	return true;
    }    
    
    // displays the current notes attached to the sector and allows adding additional notes to the
    // sector.
    private void displayNotes() {

		// build the gui for the dialog box
		JTextArea tanotes = new JTextArea(5, 30);
		JLabel addnote = new JLabel("Add a new note: ");
		newnote = new JTextArea(2, 30);
		JLabel label = new JLabel("Notes:");
		
		tanotes.setLineWrap(true);
		newnote.setLineWrap(true);
		tanotes.setWrapStyleWord(true);
		newnote.setWrapStyleWord(true);
		tanotes.setEditable(false);
    		    	
    	if (noteshash.containsKey(tempsector)) {
    		// if notes already exist, sort and initialise the text area
    		notelist = (Hashtable)noteshash.get(tempsector);
    		
    		// sort the notes by date
    		String[] dates = new String[notelist.size()];
    		int j = 0;
    		for (Enumeration i = notelist.keys(); i.hasMoreElements();) {
    			dates[j] = (String)i.nextElement();
    			System.out.print(j + " " + dates[j] + " ");
    			j++;
    		}
    		dates = sortNotes(dates);
    		
    		for (int i = 0; i < dates.length; i++) {
    			tanotes.append(dates[i] + ": ");
    			tanotes.append((String)notelist.get(dates[i]) + "\n");
    		}
    	} else {
    		notelist = new Hashtable();
    	}

    		
		box = Box.createVerticalBox();
		box.add(label);
		box.add(new JScrollPane(tanotes));
		box.add(addnote);
		box.add(new JScrollPane(newnote));
		
		Box buttonbox = Box.createHorizontalBox();
		JButton ok = new JButton("OK");
		JButton cancel = new JButton("Cancel");
		ok.addActionListener(this);
		cancel.addActionListener(this);
		buttonbox.add(ok);
		buttonbox.add(cancel);
		
		box.add(buttonbox);
		
		noteframe = new JFrame("Display Notes");
		noteframe.getContentPane().add(box);
		noteframe.setSize(300, 200);
		noteframe.setVisible(true);
		noteframe.setVisible(true);

    }
    
    // sorts the notes hashtable according to the date
    private String[] sortNotes(String[] noteDates) {
    	
    	for (int i = 0; i < noteDates.length; i++) {
    		 for (int j=i+1; j < noteDates.length; j++) {
      		 	if (noteDates[i].compareTo(noteDates[j]) > 0) {
          			String temp = noteDates[j]; 
          			noteDates[j]=noteDates[i]; 
          			noteDates[i]=temp;
          		}
          	}
        }
        
        return noteDates;
    }
    
    // adds the note the user entered in the Display Notes dialog
    private void addNote() {
    	String snote = newnote.getText();
    	if (snote != "") {
    		
    		// get the current date/time to attach to the note
    		SimpleDateFormat sdf = new SimpleDateFormat("yyyy.MM.dd 'at' HH:mm:ss z");
    		String date = sdf.format(new Date());
    		notelist.put(date, snote);
    		
    		LList tempnotes = Lisp.NIL;
    		LList temp;
    		for (Enumeration i = notelist.keys(); i.hasMoreElements();) {
    			String next = (String)i.nextElement();
    			temp = Lisp.NIL;
    			temp = Lisp.cons(notelist.get(next), temp);
    			temp = Lisp.cons(next, temp);
    			tempnotes = Lisp.cons(temp, tempnotes);
    		}
    		
    		sendConstraint(new PatternAssignment(Lisp.list(Symbol.intern("note"), Symbol.intern(tempsector)), tempnotes));
    	}   
    	noteframe.setVisible(false); 
    }
		
}